#include "nmea_parser.h"
#include <string.h>
#include <stdlib.h>

#define NMEA_TUD_KNOTS      (1.852)         /**< Knots, kilometer / NMEA_TUD_KNOTS = knot */

class DestructiveParser {

char _null_char[1];

char *m_str;

public:
       
    DestructiveParser(char *str) : m_str(str) { _null_char[0]=0; }

bool Valid() const { 
	return m_str!=0; 
    } 

char *Next()
{
    if(m_str == 0)
     return &_null_char[0];
     
	char *str = m_str;

	if(*m_str != 0)
		{
		char *n = strchr(m_str, ',');
		if(n != NULL)
			{
			*n=0;
			m_str = n + 1;
			}
		else
			{
			m_str = 0; 
			}
		}
		else
		{
        m_str = 0;
        }

	return str;
}

char *String() { return Next(); }

char *String(unsigned int & data, unsigned int mask) { 
      char *str = Next();
      
      if(str)
        data |= mask;
      
      return str;
}

char Char() { return Next()[0]; }

char Char(unsigned int & data, unsigned int mask) { 
      char *str = Next();
      
      if(str[0]!=0)
        data |= mask;
      
      return str[0];
     }

double Double(double _def) { 
	char *str = Next();
	if(str[0]!=0)
		return atof(str); 
	else
		return _def;
}

int Int(int _def) { 
	char *str = Next();
	if(str[0]!=0)
		return atoi(str); 
	else
		return _def;
}

template<class FloatType>
bool Double(FloatType & dest, unsigned int & data, unsigned int mask) { 
	char *str = Next();
	if(str[0]!=0)
	{
		dest = atof(str); 
		data |= mask;
		return true;
    }
	else
	{
        return false;
    }
}

template<class IntType>
bool Int(IntType & dest, unsigned int & data, unsigned int mask) { 
	char *str = Next();
	if(str[0]!=0)
	{
		dest = atoi(str); 
		data |= mask;
		return true;
    }
	else
	{
        return false;
    }
}
};

struct nmea_context {
       int last_param;
       int params[NMEA_MAXSAT*4+1];
       
       nmea_context() : last_param(0) { }
       };

nmea_parser::nmea_parser()
{
    valid = 0;
    updated = 0;
	sat_inview = 0;
	sat_inuse = 0;
    sat_info = 0;
	ctx = new nmea_context;
}

nmea_parser::~nmea_parser()
{ 
  delete ctx;
}

static int hex2char(char c)
{
  return ((c>='0')&&(c<='9')) ? (c-'0') :
         ((c>='a')&&(c<='f')) ? (c-'a'+10) :
         ((c>='A')&&(c<='F')) ? (c-'A'+10) : 
                              0xFF;
}

static double nmea_ndeg2degree(double val)
{
    double deg = ((int)(val / 100));
    val = deg + (val - deg * 100) / 60;
    return val;
}

static void time_to_nmeatime(nmea_parser & t, double time)
{
t.hour = (int)(time) / 10000;
t.min = ((int)(time) / 100) % 100;
t.sec = (int)(time) % 100;
t.msec = (int) ((time - (int)(time) ) * 1000);
}

static bool checksum(const char *str)
{
     const char *delim = strchr(str,'*');
     if(!delim)
      return false;
     unsigned char accum = 0;
//     printf("%s: %u bytes long\n", str, (long) delim - (long) str );
     while(str!=delim)
         {
//         putchar(*str);
         accum ^= *str;
         str++;
         }
//     printf(" %x %s\n", accum, delim);
     unsigned char sum = (hex2char(delim[1])<<4)+(hex2char(delim[2]));
     
     if(sum!=accum)
     {
//  fprintf("wrong checksum: is %u, should be %u\n", accum, sum);
     }
     
     return sum == accum;
}

/// Parse a new line (is destructive.. the line will be destroyed)
bool nmea_parser::parse(char *str)
{
    if(str[0]!='$')
       return false;

//    printf(str);
       
    if(!checksum(str+1))
       return false;    
       
    char *delim = strchr(str,'*');
    if(!delim)
     return false;
    *delim=0;
    
    DestructiveParser in(str);
    
     //    printf("begin: %x\n", updated);
    
	char *head = in.Next();

	if(!strcmp(head, "$GPGSA"))
		{
		mode = in.Char();	   // Tipo di selezione del tipo di rilevamento (2D o 3D): M=Manuale; A=Automatico
		in.Int(fix, updated, NMEA_FIX);	   // Tipo di rilevamento: 2=2D (bidimensionale); 3=3D (tridimensionale)
		int i;

		for(i=0;i<NMEA_MAXSAT;i++)
			sat[i].in_use = 0;

        sat_inuse = 0;
        updated |= NMEA_SAT_IN_USE;
		for(i=0;i<12;i++)
				{
				int prn = in.Int(-1);
				if(prn!=-1)
					{
                    sat_inuse++;
					for(int j=0;j<NMEA_MAXSAT;j++)
						if(sat[i].id == prn)
							sat[i].in_use = 1;
					}
				}
		in.Double(PDOP, updated, NMEA_PDOP);
        in.Double(HDOP, updated, NMEA_HDOP);
        in.Double(VDOP, updated, NMEA_VDOP);
		}
	else
	if(!strcmp(head, "$GPGGA"))
		{
		double time = in.Double(-1.0);	// Ora UTC  - GPS
		double latitude = in.Double(0.0);
		char latituden = in.Char();
		double longitude = in.Double(0.0);
		char longituden = in.Char();
		in.Int(sig, updated, NMEA_SIG);  // Qualit del rilevamento GPS: 0=non valido; 1=GPS; 2=DGPS
		in.Int(sat_inview, updated, NMEA_SATINVIEW); // Numero di satelliti in vista (teorica, non necessariamente in uso o ricevuti)
		in.Double(HDOP, updated, NMEA_HDOP);
		double height = in.Double(-10000); // Altitudine dell'antenna GPS relativa al livello medio del mare (geoide)
		char *heightn = in.Next();
		char *heightge = in.Next();
		char *heightgen = in.Next(); 
		char *dgpstime = in.Next(); // Tempo in secondi dall'ultimo aggiornamento DGPS
		in.Int(dgps_prn, updated, NMEA_DGPS); // Identificatore della stazione DGPS (0000-1023)
		// char *checksum =	Next();

		if(height > -10000)
			altitude = height;

		time_to_nmeatime(*this, time);

		if ((latitude != 0.0) && (longitude != 0))
			{
			this->latitude = nmea_ndeg2degree( latitude );
			if(latituden != 'N')
					this->latitude = - this->latitude;
		
			this->longitude = nmea_ndeg2degree( longitude );

			if(longituden != 'E')
					this->longitude = - this->longitude;
			updated |= NMEA_LATLON;
			}
			else
			{
            }

		}
	else
	if(!strcmp(head, "$GPRMC"))
		{
		double time = in.Double(-1);	// Ora UTC-GPS
		char status = in.Char(); // Stato: A=Active, Attivo; V=Void, Nullo (per ricordarne il significato:  come una freccia in su/gi)
		double latitude = in.Double(0.0);
		char latituden = in.Char();
		double longitude = in.Double(0.0);
		char longituden = in.Char();
		double kspeed = in.Double(-1.0);	// Velocit al suolo, in nodi
		direction = in.Double(-1.0);	// Track Made Good - Direzione di movimento, in gradi reali
		int day = in.Int(-1);          //  date from 1/1/2000

		time_to_nmeatime(*this, time);

		if('A' ==status)
		{
			/*
			if(NMEA_SIG_BAD == sig)
				sig = NMEA_SIG_MID;
			if(NMEA_FIX_BAD == fix)
				fix = NMEA_FIX_2D;
			*/
		}
		else if('V' == status)
		{
			sig = NMEA_SIG_BAD;
			fix = NMEA_FIX_BAD;
		}

		if ((latitude != 0.0) && (longitude != 0))
			{
			this->latitude = nmea_ndeg2degree( latitude );
			if(latituden != 'N')
					this->latitude = - this->latitude;

			this->longitude = nmea_ndeg2degree( longitude );

			if(longituden != 'E')
					this->longitude = - this->longitude;
			updated |= NMEA_LATLON;
			}

		if(kspeed>=0.0)
		{
			speed = NMEA_TUD_KNOTS * kspeed;
			updated |= NMEA_SPEED;
        }
		
		if(day!=-1)
		{
          year = day % 100;
          month = (day/100) % 100;
          mday = (day/10000);
          updated |= NMEA_DATE;
        }
		
		}
	else
	if(!strcmp(head, "$GPGSV"))
		{
		int nmsg = in.Int(-1); // numero messaggi nella catena
		int n = in.Int(-1);	// numero del messaggio nella catena 1..nmsg
		int count = in.Int(-1);	// numero di satelliti nella catena
		
		sat_inview = count;
		
     //		printf("Msg %u/%u\n", n, nmsg);
		
		if(n == 1) ctx->last_param = 0;
		
  	    do {
            ctx->params[ctx->last_param] = in.Int(-1);
//            printf("\t%u", ctx->params[ctx->last_param]);
            ctx->last_param++;
        }while(in.Valid());
//                printf("\n");
        ctx->params[ctx->last_param] = -1; // additional extra signal
        
        if(n == nmsg)
        {
         int slot = 0;
         updated |= NMEA_SATINFO;
//         printf("parse %u elements: should be %u sats\n", ctx->last_param, count);
         for(int i=0;i<ctx->last_param;i+=4)
         {
			sat[slot].id = ctx->params[i+0];                 
			sat[slot].elv = ctx->params[i+1];
			sat[slot].azimuth = ctx->params[i+2];
			sat[slot].sig = ctx->params[i+3];
//     		printf("[%u] id:%u sig:%u\n", slot, sat[slot].id, sat[slot].sig);
			slot++;
         }
         sat_info = slot;
        }
		
		}


  valid |= updated;
}
