#include <iostream>
#include "kalman.h"
#include <stdlib.h>
#include <fstream>

struct data {
  double time;
  double lat;
  double lon;
  double speed;
  double heading;
  double yawrate1;
  double yawrate2;
};

/** ESEMPIO DI KALMAN
 *  STATO:
 *   HEADING x0
 *   YAWRATE x1
 *   BIAS1   x2
 *   BIAS2   x3
 * 
 *  heading = heading + yawrate * t; 
 *  x0 = x0 + t * x1
 *  yawrate = yawrate + process_noise
 *  x1 = x1
 *  x2 = x2
 *  x3 = x3
 * 
 *  OSSERVAZIONE:
 *   HEADING (GPS) z0
 *   YAWRATE1      z1
 *   YAWRATE2      z2
 * 
 *  heading = heading + noise(speed)
 *  z0 = x0
 *  yawrate1 = yawrate + bias1 + noise
 *  z1 = x1 + x2
 *  yawrate2 = yawrate + bias2 + noise
 *  z2 = x1 + x3
 * */

class rcompare {
  boost::numeric::ublas::vector<double> e;
public:
  rcompare() : e(3) { }
  
  boost::numeric::ublas::vector<double> operator()(const boost::numeric::ublas::vector<double> & z, const boost::numeric::ublas::vector<double> & s)
  {
     e(0) = z(0) - s(0);
     if(e(0) > M_PI)
       e(0) -= 2*M_PI;
     if(e(0) < -M_PI)
       e(0) += 2*M_PI;
    
     e(1) = z(1) - s(1);
     e(2) = z(2) - s(2);
     
     
     return e;
  }
};

int main(int argc, char *argv[])
{
  
  kalman_filter<double, boost::numeric::ublas::matrix<double> > filter(4);
  linearized_predict<boost::numeric::ublas::matrix<double>, boost::numeric::ublas::matrix<double>, boost::numeric::ublas::matrix<double> > predict(4);
  linearized_observe<boost::numeric::ublas::matrix<double>, boost::numeric::ublas::matrix<double> > observe(4, 3);
  
  boost::numeric::ublas::vector<double> initial_state(4);
  boost::numeric::ublas::scalar_matrix<double> initial_P(4,4, 1.0);
  
  initial_state(0) = 0.0;
  initial_state(1) = 0.0;
  initial_state(2) = 0.0;
  initial_state(3) = 0.0;
  
  filter.set(initial_state, initial_P);

  
  predict.A = boost::numeric::ublas::zero_matrix<double>(4,4);
  predict.A(0,0) = 1.0;
  predict.A(0,1) = 0.0; // t
  predict.A(1,1) = 1.0;
  predict.A(2,2) = 1.0;
  predict.A(3,3) = 1.0;
  
  predict.Q = boost::numeric::ublas::zero_matrix<double>(4,4);
  predict.Q(0,0) = 0.0001;  // rumore di processo sull'heading
  predict.Q(1,1) = 0.01;   // rumore di processo sullo yawrate
  predict.Q(2,2) = 0.00001; // rumore di processo sul bias1
  predict.Q(3,3) = 0.00001; // rumore di processo sul bias2
    
  observe.H = boost::numeric::ublas::zero_matrix<double>(3,4);
  observe.H(0,0) = 1.0; 
  observe.H(1,1) = 1.0; // yawrate
  observe.H(1,2) = 1.0; // bias1
  observe.H(2,1) = 1.0; // yawrate
  observe.H(2,3) = 1.0; // bias2
  
  observe.R = boost::numeric::ublas::zero_matrix<double>(3,3);
  
  observe.R(0,0) =  0.0; // dipende dalla velocita'
  observe.R(1,1) =  2.48031e-05;  // varianza rumore
  observe.R(2,2) =  1.37115e-05;  // varianza rumore
    
  std::ifstream in(argv[1]);
  double t0 = 0.0;
  while(in)
  {
    data n;
    in >> n.time >> n.lat >> n.lon >> n.speed >> n.heading >> n.yawrate1 >> n.yawrate2;
    if(in)
    {
      boost::numeric::ublas::vector<double> z(3);
      predict.A = boost::numeric::ublas::zero_matrix<double>(4,4);
      predict.A(0,1) = n.time - t0;
  
      filter.predict(predict);
      
      z(0) = n.heading;
      z(1) = n.yawrate1;
      z(2) = n.yawrate2;
      
      observe.R(0,0) = (n.speed  < 1.0) ? 1000.0 : 0.01;  // dipende dalla velocita'
      
      filter.observe(observe, z, rcompare());
      
      // std::cout << filter.state() << std::endl;
      std::cout << filter.state()(0) << ' ' << n.heading << std::endl;
  
    t0 = n.time;
    }
  }
  
 
return 0;
}