#include <iostream>
#include "kalman.h"
#include <stdlib.h>
#include <fstream>

struct data {
  double time;
  double lat;
  double lon;
  double speed;
  double heading;
  double yawrate1;
  double yawrate2;
};


class rcompare {
  boost::numeric::ublas::vector<double> e;
public:
  rcompare() : e(3) { }
  
  boost::numeric::ublas::vector<double> operator()(const boost::numeric::ublas::vector<double> & z, const boost::numeric::ublas::vector<double> & s)
  {
     e(0) = z(0) - s(0);
     if(e(0) > M_PI)
       e(0) -= 2*M_PI;
     if(e(0) < -M_PI)
       e(0) += 2*M_PI;
    
     e(1) = z(1) - s(1);
     e(2) = z(2) - s(2);
     
     
     return e;
  }
};

/** ESEMPIO DI KALMAN
 *  STATO:
 *   X       x0
 *   Y       x1
 *   HEADING x2
 *   SPEED   x3
 * 
 *  heading = heading + yawrate * t; 
 *  x0 = x0 + t * x1
 *  yawrate = yawrate + process_noise
 *  x1 = x1
 *  x2 = x2
 *  x3 = x3
 * 
 *  OSSERVAZIONE:
 *   X 		  z0
 *   Y            z1
 *   HEADING      z2
 * 
 *  heading = heading + noise(speed)
 *  z0 = x0
 *  yawrate1 = yawrate + bias1 + noise
 *  z1 = x1 + x2
 *  yawrate2 = yawrate + bias2 + noise
 *  z2 = x1 + x3
 * */


/** ESEMPIO DI KALMAN
 *  STATO:
 *   HEADING x0
 *   YAWRATE x1
 *   BIAS1   x2
 *   BIAS2   x3
 * 
 *  heading = heading + yawrate * t; 
 *  x0 = x0 + t * x1
 *  yawrate = yawrate + process_noise
 *  x1 = x1
 *  bias* = bias* + process_noise
 *  x2 = x2
 *  x3 = x3
 * 
 *  OSSERVAZIONE:
 *   HEADING (GPS) z0
 *   YAWRATE1      z1
 *   YAWRATE2      z2
 * 
 *  heading = heading + noise(speed)
 *  z0 = x0
 *  yawrate1 = yawrate + bias1 + noise
 *  z1 = x1 + x2
 *  yawrate2 = yawrate + bias2 + noise
 *  z2 = x1 + x3
 * */



int main(int argc, char *argv[])
{
  
  kalman_filter<double, boost::numeric::ublas::matrix<double> > filter(4);
  linearized_predict<boost::numeric::ublas::matrix<double>, boost::numeric::ublas::matrix<double>, boost::numeric::ublas::matrix<double> > predict(4);
  linearized_observe<boost::numeric::ublas::matrix<double>, boost::numeric::ublas::matrix<double> > observe(4, 3);
  
  linearized_observe<boost::numeric::ublas::matrix<double>, boost::numeric::ublas::matrix<double> > observe_outage(4, 2);
  
  boost::numeric::ublas::vector<double> initial_state(4);
  boost::numeric::ublas::scalar_matrix<double> initial_P(4,4, 1.00);
  
  initial_state(0) = 0.0;
  initial_state(1) = 0.0;
  initial_state(2) = 0.0;
  initial_state(3) = 0.0;
  
  filter.set(initial_state, initial_P);

  
  predict.A = boost::numeric::ublas::zero_matrix<double>(4,4);
  predict.A(0,0) = 1.0;
  predict.A(0,1) = 0.0; // t
  predict.A(1,1) = 1.0;
  predict.A(2,2) = 1.0;
  predict.A(3,3) = 1.0;
  
//   std::cout << predict.A << std::endl;	
  
  predict.Q = boost::numeric::ublas::zero_matrix<double>(4,4);
  predict.Q(0,0) = 0.000000001;  // rumore di processo sull'heading
  predict.Q(1,1) = 0.0001;       // rumore di processo sullo yawrate: dipende dal tempo (al quadrato) dal delta_yaw_rate_max (0.00001 sembra un buon valore)
  predict.Q(2,2) = 0.000000001;  // rumore di processo sul bias1
  predict.Q(3,3) = 0.000000001;  // rumore di processo sul bias2
    
  observe.H = boost::numeric::ublas::zero_matrix<double>(3,4);
  observe.H(0,0) = 1.0; 
  observe.H(1,1) = 1.0; // yawrate
  observe.H(1,2) = 1.0; // bias1
  observe.H(2,1) = 1.0; // yawrate
  observe.H(2,3) = 1.0; // bias2
  
  observe.R = boost::numeric::ublas::zero_matrix<double>(3,3);
  
  observe.R(0,0) =  0.0; // dipende dalla velocita'
  observe.R(1,1) =  3.761457936e-05;  // varianza rumore
  observe.R(2,2) =  1.405881724e-05;  // varianza rumore
  
  
  observe_outage.H = boost::numeric::ublas::zero_matrix<double>(2,4);
  observe_outage.H(0,1) = 1.0; // yawrate
  observe_outage.H(0,2) = 1.0; // bias1
  observe_outage.H(1,1) = 1.0; // yawrate
  observe_outage.H(1,3) = 1.0; // bias2    

  observe_outage.R = boost::numeric::ublas::zero_matrix<double>(2,2);
  
  observe_outage.R(0,0) =  2.48031e-05;  // varianza rumore
  observe_outage.R(1,1) =  1.37115e-05;  // varianza rumore

  std::ifstream in(argv[1]);
  double t0 = 0.0;
  double yaw0 = 0.0;
  int count = 0;
  while(in)
  {
    data n;
    
    in >> n.time >> n.lat >> n.lon >> n.speed >> n.heading >> n.yawrate1 >> n.yawrate2;
    if(in && (n.speed > 0.1) )
    {
      
      predict.A(0,1) = n.time - t0;
      
      if(count < 10)
      {
      predict.Q(0,0) = 0.1*0.1;  // rumore di processo sull'heading
      predict.Q(1,1) =  0.0001;       // rumore di processo sullo yawrate: dipende dal tempo (al quadrato) dal delta_yaw_rate_max (0.00001 sembra un buon valore)
      predict.Q(2,2) = 0.1*0.1;  // rumore di processo sul bias1
      predict.Q(3,3) = 0.1*0.1;  // rumore di processo sul bias2
      }
      else
      {
      predict.Q(0,0) = 0.000000001;  // rumore di processo sull'heading
      predict.Q(1,1) = 0.0001;       // rumore di processo sullo yawrate: dipende dal tempo (al quadrato) dal delta_yaw_rate_max (0.00001 sembra un buon valore)
      predict.Q(2,2) = 0.000000001;  // rumore di processo sul bias1
      predict.Q(3,3) = 0.000000001;  // rumore di processo sul bias2
      }
      
      
      n.heading = -n.heading;
      
      if(n.heading > M_PI)
	n.heading -= M_PI*2.0;
      if(n.heading < -M_PI)
	n.heading += M_PI*2.0;
      
//       std::cout << predict.A << std::endl;	
  
      filter.predict(predict);
      
      //if(n.time < 500.0)
      if(1)
      {
      boost::numeric::ublas::vector<double> z(3);
      z(0) = n.heading;
      z(1) = n.yawrate1;
      z(2) = n.yawrate2;
      
      observe.R(0,0) = (n.speed > 2.0) ? (0.02*0.02) : 100.0;  // dipende dalla velocita', ma in che modo? comunque sopra 0.02*0.02 funziona male
      
      filter.observe(observe, z, rcompare());
      }
      else
      {
      boost::numeric::ublas::vector<double> z(2);
      z(0) = n.yawrate1;
      z(1) = n.yawrate2;
      
      filter.observe(observe_outage, z, linear_compare<double>());
      }
      
      if(filter.state()(0) > M_PI)
	filter.state()(0) -= M_PI*2.0;
      if(filter.state()(0) < -M_PI)
	filter.state()(0) += M_PI*2.0;
      
      // std::cout << filter.CovarianceMatrix() << std::endl;
            
      std::cout << n.time << ' ' << n.speed << ' ' << n.heading << ' ' << filter.state()(0) << ' ' << filter.state()(1) << ' ' << n.yawrate1 - filter.state()(2) << ' ' << n.yawrate2 - filter.state()(3) << ' ' <<  filter.state()(2) << ' ' <<  filter.state()(3) << ' ' << ( n.heading - yaw0 ) / ( n.time - t0) << ' ' << std::endl;
      yaw0 = n.heading;
  
    t0 = n.time;
    count++;
    }
  }
  
 
return 0;
}