#include <iostream>
#include "kalman.h"
#include <stdlib.h>
#include <fstream>

double _frand()
{
  return (2.0 * (double) rand() / (double) RAND_MAX ) -1.0;
}

// var = a*a/3 -> a = sqrt(3*var)
double frand(double v)
{
  return _frand() * std::sqrt( 3.0 * v );
}

class motion_predict: public linearized_predict<>
{
  boost::numeric::ublas::vector<double> new_state;
public:
  motion_predict() : linearized_predict<>(6),
  new_state(6)
  {
    A = boost::numeric::ublas::zero_matrix<double>(6,6);

    // process noise
    Q = boost::numeric::ublas::zero_matrix<double>(6,6);
    Q(0,0) = 0.000000001;  //x no process noise
    Q(1,1) = 0.000000001;  //y no process noise
    Q(2,2) = 0.000000001;  //v no process noise
    Q(3,3) = 0.1;          //a process noise in worst case
    Q(4,4) = 0.0000000001; //yr process noise in worst case
    Q(5,5) = 0.000000001;  //y no process noise
    
  }
  
  void update(const boost::numeric::ublas::vector<double> & x)
  {
    // le derivate di f:
    
    A(0,0) = 1.0;
    A(0,2) = cos(x(5)) * 0.1;
    A(0,5) = - x(2) * sin(x(5)) * 0.1;
    
    A(1,0) = 1.0;
    A(1,2) = sin(x(5)) * 0.1;
    A(1,5) = x(2) * cos(x(5)) * 0.1;
    
    A(2,2) = 1.0;
    A(2,3) = 0.1;
    
    A(3,3) = 1.0;
    
    A(4,4) = 1.0;
    
    A(5,5) = 1.0;
    A(5,4) = 0.1;
    
  }
  
  const boost::numeric::ublas::vector<double> & f(const boost::numeric::ublas::vector<double> & x)
  {
    new_state(0) = x(0) + x(2) * cos(x(5)) * 0.1;
    new_state(1) = x(1) + x(2) * sin(x(5)) * 0.1;
    new_state(2) = x(2) + x(3) * 0.1;
    new_state(3) = x(3);
    new_state(4) = x(4);
    new_state(5) = x(5) + x(4) * 0.1;
    return new_state;
  }
  
  
};

/** esempio di un sistema del tipo:
 * 
 *  x' = x + Vt * cos(theta) [0]
 *  y' = y + Vt * sin(theta) [1]
 *  V' = V + at [2]
 *  a' = a; [3]
 *  yaw_rate = yaw_rate [4]
 *  theta = theta + yaw_rate * t [5]
 * */
int main()
{
  static const int niter = 1000;

  std::ifstream in("speed2.dat");
  extended_kalman_filter<double, boost::numeric::ublas::matrix<double> > filter(6);
  motion_predict predict;
  linearized_observe<boost::numeric::ublas::matrix<double>, boost::numeric::ublas::matrix<double> > observe(4, 6);
  
  boost::numeric::ublas::vector<double> initial_state(6);
  boost::numeric::ublas::matrix<double> initial_P(6,6);
  static double s0_var = 2.5*2.5; // 2.5m sigma
  static double s2_var = 0.001809624248; // accel (?)
  static double s4_var = 0.005997 * 0.005997; // yawrate 30 gradi
  /// ground thruth
  boost::numeric::ublas::vector<double> gt(6);

  gt(0) = 0.0; // frand(s0_var);
  gt(1) = 0.0; // frand(s0_var);
  gt(2) = 0.00;
  gt(3) = 0.0; // frand(s2_var);
  gt(4) = 2.0*M_PI / (0.1 * niter); // + frand(s4_var);
  gt(5) = 0.0;
  
  
  initial_state = gt;
     
  initial_P = boost::numeric::ublas::zero_matrix<double>(6,6);
  initial_P(0,0) = s0_var;
  initial_P(1,1) = s0_var;
  initial_P(2,2) = 0.01;
  initial_P(3,3) = s2_var;
  initial_P(4,4) = s4_var;
  initial_P(5,5) = 0.01;
  
  filter.set(initial_state, initial_P);
    
  // 
  observe.H = boost::numeric::ublas::zero_matrix<double>(4,6);
  observe.H(0,0) = 1.0; // z(0) vedo X x(0)
  observe.H(1,1) = 1.0; // z(1) vedo Y x(1)
  observe.H(2,4) = 1.0; // z(2) vedo yawrate x(4)
  observe.H(3,3) = 1.0; // z(3) vedo a x(3)
  
  // rumore di osservazione
  observe.R = boost::numeric::ublas::zero_matrix<double>(4,4); 
  observe.R(0,0) = s0_var; 
  observe.R(1,1) = s0_var; 
  observe.R(2,2) = s4_var; 
  observe.R(3,3) = s2_var; 
  
//   std::cout << "system: " << predict.A << std::endl;

  
  double varaccel = 0.0;
  double v0 = 0.0;
  double a0 = 0.0;
  double s1 = 0.0;
  double v1 = 0.0;

  /*
  double truex = 0.0;
  double truey = 0.0;
  double trueyaw = 0.0;
  double trueyawrate = 0.0;
  double truespeed = 0.01;
  double trueaccel = 0.0;
  */
  
  
  
  
  for(int count = 0;count<niter;count++)
    {
      double accel;
      double tmp, truespeed;
      
      in >> tmp >> truespeed;
      
//       std::cout << truespeed << '\n';
            
      boost::numeric::ublas::vector<double> z(4);
      

      gt(4) = 2.0*M_PI / (0.1 * niter); // yawrate
      gt(3) = (truespeed - gt(2)) / 0.1; // accel
      gt = predict.f(gt);
      
      filter.predict(predict);
      
      z(0) = gt(0) + frand(s0_var);
      z(1) = gt(1) + frand(s0_var);
      z(2) = gt(4) + frand(s4_var); 
      z(3) = gt(3) + frand(s2_var); 
      
      filter.linear_observe(observe, z, linear_compare<double>());
      // 1 2 3 x 
      // 4 5 6 y
      // 7 8 9 yawrate
      // 10 11 speed
      // 12 13 14 accel
      // 15 16 yaw
      std::cout << gt(0) << ' ' << z(0) << ' ' << filter.state()(0) << ' '
		<< gt(1) << ' ' << z(1) << ' ' << filter.state()(1) << ' '
		<< gt(4) << ' ' << z(2) << ' ' << filter.state()(4) << ' ' 
	        << gt(2) << ' ' << filter.state()(2) << ' '
	        << gt(3) << ' ' << z(3) << ' ' << filter.state()(3) << ' '
	        << gt(5) << ' ' << filter.state()(5) << ' '	
	        << std::endl;
      
    }
  
return 0;
}