#include <iostream>
#include "kalman.h"
#include <stdlib.h>
#include <fstream>

double _frand()
{
  return (2.0 * (double) rand() / (double) RAND_MAX ) -1.0;
}

// var = a*a/3 -> a = sqrt(3*var)
double frand(double v)
{
  return _frand() * std::sqrt( 3.0 * v );
}


/** esempio di un sistema del tipo:
 * 
 *  S' = S + Vt 
 *  V' = V + at
 *  a' = a
 *  z = S + n
 *  a = a + n
 * */
int main()
{
  std::ifstream in("speed2.dat");
  kalman_filter<double, boost::numeric::ublas::matrix<double> > filter(3);
  linearized_predict<boost::numeric::ublas::matrix<double>, boost::numeric::ublas::matrix<double>, boost::numeric::ublas::matrix<double> > predict(3);
  linearized_observe<boost::numeric::ublas::matrix<double>, boost::numeric::ublas::matrix<double> > observe(3, 2);
  
  boost::numeric::ublas::vector<double> initial_state(3);
  boost::numeric::ublas::matrix<double> initial_P(3,3);
  static double s0_var = 8.0*8.0;
  static double s2_var = 0.2*0.2;
  
  initial_state(0) = frand(s0_var);
  initial_state(1) = 0.00;
  initial_state(2) = frand(s2_var);
  
  initial_P = boost::numeric::ublas::zero_matrix<double>(3,3);
  initial_P(0,0) = s0_var;
  initial_P(1,1) = 0.1;
  initial_P(2,2) = s2_var;
  
  filter.set(initial_state, initial_P);
  
  double truevalue = 0.0;
  double truespeed = 0.01;
  double trueaccel = 0.0;
  
  predict.A = boost::numeric::ublas::zero_matrix<double>(3,3);
  predict.A(0,0) = 1.0;
  predict.A(0,1) = 0.1; // 0.1 sec
  predict.A(0,2) = 0.5*0.1*0.1; // 0.1 sec
  predict.A(1,1) = 1.0; 
  predict.A(1,2) = 0.1; // 0.1 sec
  predict.A(2,2) = 1.0; 
  
  // process noise
  predict.Q = boost::numeric::ublas::zero_matrix<double>(3,3);
  predict.Q(0,0) = 0.000000001;  // no process noise
  predict.Q(1,1) = 0.000000001;  // no process noise
  predict.Q(2,2) = 0.01;     // process noise in worst case
  
  // 
  observe.H = boost::numeric::ublas::zero_matrix<double>(2,3);
  observe.H(0,0) = 1.0; // vedo S al 100%
  observe.H(1,2) = 1.0; // vedo a
  
  // rumore di osservazione
  observe.R(0,0) = s0_var; // high observation noise on position 2.5metri
  observe.R(1,1) = s2_var; //0.2*0.2; // high observation noise on acceleration
  
//   std::cout << "system: " << predict.A << std::endl;
  
  double x = 0.0; 
  double p = 0.0;
  static int niter = 1000;
  
  double varaccel = 0.0;
  double v0 = 0.0;
  double a0 = 0.0;
  double s1 = 0.0;
  double v1 = 0.0;
  
  for(int count = 0;count<niter;count++)
    {
      double accel;
      double tmp, truespeed;
      
      in >> tmp >> truespeed;
      
//       std::cout << truespeed << '\n';
            
      boost::numeric::ublas::vector<double> z(2);
      filter.predict(predict);
      
      truevalue += v0 * 0.1 + a0 * 0.1 * 0.1 * 0.5;
      trueaccel = (truespeed - v0) / 0.1;
      
      varaccel += (trueaccel - a0)*(trueaccel - a0);
      
      a0 = trueaccel;
      v0 = truespeed;
      
      z(0) = truevalue + frand(s0_var);
      z(1) = trueaccel + frand(s2_var); 
      
      s1 += v1 * 0.1 + z(1) * 0.1 * 0.1 * 0.5;
      v1 += z(1) * 0.1;
      
      filter.observe(observe, z, linear_compare<double>());
      
      std::cout << truevalue << ' ' << z(0) << ' ' << filter.state()(0) << ' ' 
	        << truespeed << ' ' << filter.state()(1) << ' '
	        << trueaccel << ' ' << z(1) << ' ' << filter.state()(2) << ' '	<< v1 << ' ' << s1
	        << std::endl;
      
    }
  
std::cerr << "ap variance:" << (varaccel / niter) << std::endl;
std::cerr << "ap sqrt:" << std::sqrt(varaccel / niter) << std::endl;  
  
return 0;
}