#ifndef _KALMAN_H
#define _KALMAN_H

/*
Copyright (C) 2011,2012 Paolo Medici

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/** @file kalman.h
 *  @brief this file implements Kalman Filter (KF), Extended Kalman Filter (EKF) and Unscented Kalman Filter (UKF)
 **/

// TODO: migrate to Eigen

#include <boost/numeric/ublas/vector.hpp>
#include <boost/numeric/ublas/matrix.hpp>
#include <boost/numeric/ublas/lu.hpp>
#include <boost/numeric/ublas/io.hpp>

#include "cholesky.hpp" //

#ifndef debug_kalman
# define debug_kalman false
#endif

 /** Matrix inversion routine. (@todo da spostare in libreria)
    Uses lu_factorize and lu_substitute in uBLAS to invert a matrix 
 */
template<class T>
bool invert(const boost::numeric::ublas::matrix<T>& input, boost::numeric::ublas::matrix<T>& inverse)
{
	typedef boost::numeric::ublas::permutation_matrix<std::size_t> pmatrix;

	// create a working copy of the input
	boost::numeric::ublas::matrix<T> A(input);

	// create a permutation matrix for the LU-factorization
	pmatrix pm(A.size1());

	// perform LU-factorization
	int res = boost::numeric::ublas::lu_factorize(A, pm);
	if (res != 0)
		return false;

	// create identity matrix of "inverse"
	inverse.assign(boost::numeric::ublas::identity_matrix<T> (A.size1()));

	// backsubstitute to get the inverse
	boost::numeric::ublas::lu_substitute(A, pm, inverse);

	return true;
}

/// Genera i Sigma Points partendo da x e Pxx e li proietta attraverso la policy @a f
/// @param f funzione per trasformare @a x
/// @param[out] sp  i sigma points trasformati
/// @param[in] x lo stato
/// @param[in] Pxx matrice di covarianza dello stato
/// @param gamma scaling factor gamma = sqrt(n + lambda)
template<class _Policy, class T>
void sigma_points(_Policy f, boost::numeric::ublas::vector<T> * sp, const boost::numeric::ublas::vector<T> & x, const boost::numeric::ublas::matrix<T> & Pxx, double gamma)
{
   const int n = x.size();
   boost::numeric::ublas::triangular_matrix<double, boost::numeric::ublas::lower> p(Pxx.size1(), Pxx.size2()); // dovrebbe essere quadrata
   cholesky_decompose(Pxx, p);
   
   // sigma point centrale
   sp[0] = f(x);
   
   // per ogni dimensione n
   for(int i = 0;i<n; ++i)
   {
     // sigma point
     boost::numeric::ublas::vector<T> v = gamma * boost::numeric::ublas::matrix_column< boost::numeric::ublas::triangular_matrix<double, boost::numeric::ublas::lower> > (p, i);
     sp[1+i*2] = f( boost::numeric::ublas::vector<T>(x + v) );
     sp[2+i*2] = f( boost::numeric::ublas::vector<T>(x - v) );
   }
   
   if(debug_kalman)
   {
   std::cerr << "sigma points:\n";
   for(int i = 0;i<2*n+1; ++i)
     std::cerr << i << '\t' << sp[i] << std::endl;
   }
}

/// proietta i sigma points @a xp sui sigma points @a yp attraverso la policy @a policy
template<class _Policy, class T>
void project(_Policy policy, boost::numeric::ublas::vector<T> * yp, const boost::numeric::ublas::vector<T> * xp, int n)
{
  for(int i =0;i<n;++i)
  {
    const boost::numeric::ublas::vector<T> & t = policy(xp[i]);
    yp[i] = t;
  }
}

/// dati dei sigma points yp calcola il valor medio y
/// weight are lambda/(n+lambda) for central moment, and 1/2(n + lambda) for other moment
template<class T>
void average(boost::numeric::ublas::vector<T> & y, const boost::numeric::ublas::vector<T> * yp, int n, double lambda)
{
   double scale = n+lambda;
   // pesi e fattore di normalizzazione: w0 + 2 * n * wi = 1
   double w0 = lambda / scale;	  // central moment
   double wi = 1.0 / (2.0*scale); // others moment
   
   // accumulatore stato
   y = yp[0];
   y *= w0;
   
   // per ogni dimensione
   for(int i = 0;i<n; ++i)
   {
     // compute average
     boost::numeric::ublas::vector<T> sum = yp[1+i*2] + yp[2+i*2];
     sum *= wi;
     y += sum;
   }
   
}

/// calcola la matrice di covarianza Pyy dato y e yp.
/// weight are lambda/(n+lambda) + central_boost for central moment, and 1/2(n + lambda) for other moment
template<class T>
void covariance( boost::numeric::ublas::matrix<T> & Pyy, const boost::numeric::ublas::vector<T> & y, const boost::numeric::ublas::vector<T> * yp, int n, double lambda, double central_boost)
{
   double scale = n+lambda;
   double w0 = (lambda / scale) + central_boost; // central moment
   double wi = 1.0 / (2.0*scale);                // other moment
   boost::numeric::ublas::vector<T> tmp;
   
   tmp = yp[0] - y;
   // procedo al calcolo di Pyy
   Pyy = boost::numeric::ublas::outer_prod(tmp, tmp);
//    std::cerr << "outer_prod:" << Pyy << std::endl;
   Pyy *= w0;
   
   for(int i =1;i<2*n+1;++i)
   {
     tmp = yp[i] - y;
     Pyy+= wi * boost::numeric::ublas::outer_prod(tmp, tmp);
     
   }
}

/// calcola la matrice di covarianza Pyy dato y e yp e crosscovarianza Pxy dato x e xp
template<class T>
void covariance( boost::numeric::ublas::matrix<T> & Pxy, 
		 boost::numeric::ublas::matrix<T> & Pyy, 
		 const boost::numeric::ublas::vector<T> & y, 
		 const boost::numeric::ublas::vector<T> * yp,
		 const boost::numeric::ublas::vector<T> & x, 
		 const boost::numeric::ublas::vector<T> * xp, int n, double lambda, double central_boost)
{
   double scale = n+lambda;
   double w0 = (lambda / scale) + central_boost ;
   double wi = 1.0 / (2.0*scale);
   boost::numeric::ublas::vector<T> yt, xt;

   yt = yp[0] - y;
   xt = xp[0] - x;
   // procedo al calcolo di Pyy
   Pyy = boost::numeric::ublas::outer_prod(yt, yt);
   Pyy *= w0;
//    std::cerr << "outer_prod:" << Pyy << std::endl;
   
   Pxy = boost::numeric::ublas::outer_prod(xt, yt);
   Pxy *= w0;
   
   for(int i =1;i<2*n+1;++i)
   {
     yt = yp[i] - y;
     xt = xp[i] - x;
     Pyy+= wi * boost::numeric::ublas::outer_prod(yt, yt);
     Pxy+= wi * boost::numeric::ublas::outer_prod(xt, yt);
   }
}

/// calcola y e Pyy da yp.
template<class T>
void unscent_transform(boost::numeric::ublas::vector<T> & y, boost::numeric::ublas::matrix<T> & Pyy, const boost::numeric::ublas::vector<T> * yp, int n, double lambda, double central_boost)
{
  average(y, yp, n, lambda);
  covariance(Pyy, y, yp, n, lambda, central_boost);
}


#if 0
/// calcola y, Pxy e Pyy parento da sigma points gia' calcolati
template<class _Policy>
void unscent_transform(_Policy policy, boost::numeric::ublas::vector<T> & y, boost::numeric::ublas::matrix<T> & Pxy, boost::numeric::ublas::matrix<T> & Pyy,  boost::numeric::ublas::vector<T> * xp, int n, double k)
{
     // accumulatore pesi
   boost::numeric::ublas::vector<T> * yp = new boost::numeric::ublas::vector<T>(1 + 2*n);
   
   // proietto xp su yp usando policy
   project(policy, yp, xp);
   
   average(y, yp, n, k);
   
   // calcolo la covarianza e la crosscovarianza
   covariance(Pyy, Pxy, y, yp, x, xp, n, k);
   
   delete [] yp;
}
#endif

/** funzione che trasforma x e Pxx in accordo con la funzione f */
#if 0
template<class _Policy, class T>
void unscent_transform(_Policy f, boost::numeric::ublas::vector<T> & y, boost::numeric::ublas::matrix<T> & Pyy, const boost::numeric::ublas::vector<T> & x, const boost::numeric::ublas::matrix<T> & Pxx, T k)
{
   const int n = x.size;
   const double k = kappa(n);
   
   // Sigma points
   boost::numeric::ublas::vector<T> * sp = new boost::numeric::ublas::vector<T>(1 + 2*n);
   
   sigma_points(f, sp, x, Pxx);
   
   unscent_transform(y, Pyy, sp, n,k);

   delete [] sp;
}
#endif

template<class _Predict>
class _predict_policy {
 _Predict & predict;
public:
  _predict_policy(_Predict & p) : predict(p) { }
  template<class T>
  boost::numeric::ublas::vector<T> operator()(const boost::numeric::ublas::vector<T> & x)
  {
    return predict.f(x);
  }
};

template<class _Observe>
class _observe_policy {
 _Observe & observe;
public:
  _observe_policy(_Observe & o): observe(o) { }
  template<class T>
  boost::numeric::ublas::vector<T> operator()(const boost::numeric::ublas::vector<T> & x)
  {
    return observe.h(x);
  }
};


/** Implements a linear (o linearized) kalman filter 
 * @param T precision (usually double)
 * @param _matP type of covariance matrix ( boost::numeric::ublas::matrix \< T \> )
 * */
template<class T, class _matP = boost::numeric::ublas::matrix<T> >
class kalman_filter {
protected:
  /// hidden status
 boost::numeric::ublas::vector<T> x;
 /// covariance matrix
 _matP P;
 /// Kalman gain
 boost::numeric::ublas::matrix<T> K;
 /// the identity matrix
 boost::numeric::ublas::identity_matrix<T> I;
 public:

 /// initialize the filter with a state @a size
 kalman_filter(int size) : x(size), P(size, size), K(size, size), I(size,size) { }
 kalman_filter() { }
 
 /// initialize the filter with a state @a size
 void init(int size)
 {
   x = boost::numeric::ublas::vector<T>(size);
   P = _matP(size, size);
   K = boost::numeric::ublas::matrix<T>(size, size);
   I = boost::numeric::ublas::identity_matrix<T>(size, size);
 }
 
 /// return the state size
 unsigned int size() const { return x.size(); }
 
 /// initialize the hidden status @a _x and covariance matrix @a _P
 template<class _s>
 void set(const boost::numeric::ublas::vector<T> & _x, const _s & _P)
 {
   x = _x;
   P = _P;
 }
 
//   template<class _Predict>
//   void unscented_predict (_Predict & predict)
//   {
//      cholesky_decompose(P, mC);
// 	const std::size_t XX_size = XX.size2();
// 
// 						// Create unscented distribution
// 	kappa = predict_Kappa(x_size);
// 	Float x_kappa = Float(x_size) + kappa;
// 	unscented (XX, x, X, x_kappa);
// 
// 						// Predict points of XX using supplied predict model
// 							// State covariance
// 	for (std::size_t i = 0; i < XX_size; ++i) {
// 		column(fXX,i).assign (f.f( column(XX,i) ));
// 	}
// 
// 	init_XX ();
// 						// Addative Noise Prediction, computed about center point
// 	noalias(X) += f.Q( column(fXX,0) );
//   } 
//   

/// predict without input (predict must have predict.A matrix
///  and predict.update(x) method)
/// x' = Ax
/// @see use linearized_predict
 template<class _Predict>
 void predict(_Predict & predict)
 {
   // 
//    predict.update(x);
   // stima a priori di x
   // x = predict.A * x
   x = boost::numeric::ublas::prod(predict.A, x); 
//    std::cerr << "priori: " << x << std::endl;
   // aggiornamento della covarianza a priori
   // P = predict.A * P *  boost::numeric::ublas::trans(predict.A) + predict.Q;
   P = boost::numeric::ublas::prod(predict.A, 
				   boost::numeric::ublas::matrix<T>(boost::numeric::ublas::prod(P , boost::numeric::ublas::trans(predict.A)))) + predict.Q;
 }

/// predict with input (predict must have predict.A e predict.B
///  and predict.update(x, u) method
/// x' = Ax + Bu
 template<class _Predict>
 void predict(_Predict & predict,  const boost::numeric::ublas::vector<T> & input)
 {
//    predict.update(x, input);
   // stima a priori di x
   x = predict.A * x + predict.B * input;
//    std::cerr << "priori: " << x << std::endl;
   // aggiornamento della covarianza a priori
   P = predict.A * P *  boost::numeric::ublas::trans(predict.A) + predict.Q;
 }
 
 /// simulate the future state of the system (without change it)
 /// @param x1 return the future state
 void simulate(boost::numeric::ublas::vector<T> & x1)
 {
   x1 = predict.A * x;
 }

 /// simulate the future state of the system (without change it)
 /// @param x1 return the future state
 /// @param input the input param
 void simulate(const boost::numeric::ublas::vector<T> & input, boost::numeric::ublas::vector<T> & x1)
 {
   x1 = predict.A * x + predict.B * input;
 }

/// Observe. Observe must have observe.H matrix and observe.update(x) method.
/// @param comp the observation comparting funcation. It is needed because error computing could be not linear (es circular distance). Use linear_compare as default
/// @see use linearized_observe
 template<class _Observe, class _Compare>
 void observe(_Observe & observe, const boost::numeric::ublas::vector<T> & z, _Compare comp)
 {
   // TODO: check matrix size
   if(x.size() != P.size1() || x.size() != P.size2()) 
      throw std::runtime_error("kalman, wrong status or P size");
   if(I.size1() != P.size1() || I.size2() != P.size2()) 
      throw std::runtime_error("kalman, wrong status or P size");
//    observe.update(x);
   
   boost::numeric::ublas::matrix<T> tmp3;
   boost::numeric::ublas::matrix<T> tmp2;
   
   // HPH^{T} + R
   tmp3 = boost::numeric::ublas::prod(P, boost::numeric::ublas::trans(observe.H));
   tmp2 = boost::numeric::ublas::prod(observe.H, tmp3);
   tmp2 += observe.R;
   if(debug_kalman)
    std::cerr << "tmp2: " << tmp2 << std::endl;
   
   boost::numeric::ublas::matrix<T> tmp(tmp2.size1(), tmp2.size2());
   
   invert(tmp2, tmp);
   if(debug_kalman)   
     std::cerr << "tmp: " << tmp << std::endl;
// calcolo del kalman gain:
//    K = P * boost::numeric::ublas::trans(observe.H) * tmp;
   K = boost::numeric::ublas::prod(P, boost::numeric::ublas::matrix<T>(boost::numeric::ublas::prod(boost::numeric::ublas::trans(observe.H), tmp)));
   // stima a posteriori dello stato
   // x += K * (z - observe.H * x);
   if(debug_kalman)
    std::cerr << "K gain: " << K << std::endl;
   x += boost::numeric::ublas::prod(K, comp(z, boost::numeric::ublas::prod(observe.H, x) ) );
   if(debug_kalman)
    std::cerr << "x posteriori: " << x << std::endl;
   // Stima a posteriori dell'errore di covarianza
   // P = (I - K * observe.H) * P;
   P = boost::numeric::ublas::prod(I - boost::numeric::ublas::prod(K ,observe.H), P);
   if(debug_kalman)
     std::cerr << "P Posteriori: " << P << std::endl;
 }
 
 
 /// return the inner state
 const boost::numeric::ublas::vector<T> & state() const { return x; }
 boost::numeric::ublas::vector<T> & state() { return x; }

 /// return the state covariance matrix
 const _matP & CovarianceMatrix() const { return P;} 
};

/// EKF, the extended kalman filter
/// _Predict e _Observe object must have an update method in order to renew matrixes
/// _Predict must implements also a f generic function for state update
template<class T, class _matP = boost::numeric::ublas::matrix<T> >
class extended_kalman_filter : public kalman_filter<T, _matP> {
public:
  extended_kalman_filter(int size) : kalman_filter<T, _matP>(size)
  {
  }
  
 /// first step in kalman filter, prediction. 
 /// @param _Predict must have f method and A matrix member
 template<class _Predict>
 void predict(_Predict & predict)
 {
   // 
    predict.update(this->x);
    
   // stima a priori di x
   // x = predict.A * x
   this->x = predict.f(this->x); 
//    std::cerr << "priori: " << x << std::endl;
   // aggiornamento della covarianza a priori
   // P = predict.A * P *  boost::numeric::ublas::trans(predict.A) + predict.Q;
   this->P = boost::numeric::ublas::prod(predict.A, 
				   boost::numeric::ublas::matrix<T>(boost::numeric::ublas::prod(this->P , boost::numeric::ublas::trans(predict.A)))) + predict.Q;
 }
  
 /// first step in kalman filter, prediction, with input
 /// @param _Predict must have f method and A matrix member
 template<class _Predict>
 void predict(_Predict & predict,  const boost::numeric::ublas::vector<T> & input)
 {
   predict.update(this->x, input);
   // stima a priori di x
   this->x = predict.f(this->x, input);
//    std::cerr << "priori: " << x << std::endl;
   // aggiornamento della covarianza a priori
   this->P = predict.A * this->P *  boost::numeric::ublas::trans(predict.A) + predict.Q;
 }
 
/// Observe. Observe must have observe.H matrix and observe.update(x) method.
/// @param comp is needed due to fact that error computing could be not linear (es circular distance). Use linear_compare as default
/// @see use linearized_observe
 template<class _Observe, class _Compare>
 void observe(_Observe & observe, const boost::numeric::ublas::vector<T> & z, _Compare comp)
 {
   observe.update(this->x);
   
   boost::numeric::ublas::matrix<T> tmp3;
   boost::numeric::ublas::matrix<T> tmp2;
   
   // HPH^{T} + R
   tmp3 = boost::numeric::ublas::prod(this->P, boost::numeric::ublas::trans(observe.H));
   tmp2 = boost::numeric::ublas::prod(observe.H, tmp3);
   tmp2 += observe.R;
   
//    std::cerr << "tmp2: " << tmp2 << std::endl;
   
   boost::numeric::ublas::matrix<T> tmp(tmp2.size1(), tmp2.size2());
   
   invert(tmp2, tmp);
   
//    std::cerr << "tmp: " << tmp << std::endl;
   // calcolo del kalman gain:
//    K = P * boost::numeric::ublas::trans(observe.H) * tmp;
   this->K = boost::numeric::ublas::prod(this->P, boost::numeric::ublas::matrix<T>(boost::numeric::ublas::prod(boost::numeric::ublas::trans(observe.H), tmp)));
   // stima a posteriori dello stato
   // x += K * (z - observe.H * x);
//    std::cerr << "gain: " << K << std::endl;
   this->x += boost::numeric::ublas::prod(this->K, comp(z, observe.h(this->x) ) );
//    std::cerr << "posteriori: " << x << std::endl;
   // Stima a posteriori dell'errore di covarianza
   // P = (I - K * observe.H) * P;
   this->P = boost::numeric::ublas::prod(this->I - boost::numeric::ublas::prod(this->K ,observe.H), this->P);
//    std::cerr << "P Posteriori: " << P << std::endl;
 }
 
 
 /// chiama la observe di kalman lineare. Non richiede h, non chiama la update
 template<class _Observe, class _Compare>
 void linear_observe(_Observe & observe, const boost::numeric::ublas::vector<T> & z, _Compare comp)
 {
   kalman_filter<T, _matP>::observe(observe, z, comp);
 }
 
};

/// UKF, the unscented kalman filter
template<class T, class _matP = boost::numeric::ublas::matrix<T> >
class unscented_kalman_filter: public kalman_filter<T, _matP> {
    boost::numeric::ublas::vector<T> y;
    boost::numeric::ublas::matrix<T> Pxy;
    boost::numeric::ublas::matrix<T> Pyy;
   
   // Sigma points
   boost::numeric::ublas::vector<T> * sp;
   boost::numeric::ublas::vector<T> * yp;
   
   // parametri empirici
   double alpha; ///< compreso tra 1e-4 e 1
   double beta;  ///< vale 2 per le gaussiane
   double kappa; ///< vale 0 normalmente

   // parametri calcolati
   double lambda;		///< lambda = a^2(n+k)-n
   double gamma;		///< gamma = sqrt(n + lambda)
   double central_boost;	///< boost = 1 - a^2 + b

private:

  unscented_kalman_filter(const unscented_kalman_filter & a) { }

public:

/// initialize the filter with a state @a size
unscented_kalman_filter(int size) : kalman_filter<T, _matP>(size)
{
   sp = new boost::numeric::ublas::vector<T>[1 + 2*size];
   yp = new boost::numeric::ublas::vector<T>[1 + 2*size];
   
   set_weights(1e-3, 2.0, 3.0-size);
}

unscented_kalman_filter() : kalman_filter<T, _matP>()
{
   set_weights(1e-3, 2.0, 0.0);
}

~unscented_kalman_filter() 
{
  delete [] yp;
  delete [] sp;
}
  

/// initialize the filter with a state @a size
void init(int size) 
{
  kalman_filter<T, _matP>::init(size);
  
  delete [] sp;
  delete [] yp;
   sp = new boost::numeric::ublas::vector<T>[1 + 2*size];
   yp = new boost::numeric::ublas::vector<T>[1 + 2*size];
}

/// set UKF parameters
/// NOTE: if size changes, have to be set again
void set_weights(double a, double b, double k)
{
  alpha = a;
  beta = b;
  kappa = k;
  
  lambda = alpha * alpha * (this->size() + kappa) - this->size(); // a^2(n+k)-n
  gamma = std::sqrt(this->size() + lambda);   // sqrt( n + lambda)
  central_boost = 1.0 - alpha * alpha + beta; // 1 - a^2 + b
}

 /// simulate the future state of the system (without change it)
 /// @param x1 return the future state
 template<class _Predict>
 void simulate(boost::numeric::ublas::vector<T> & x1, _Predict & predict)
 {
    const unsigned int n = this->size();
    boost::numeric::ublas::matrix<T> tmp;
    
    sigma_points(_predict_policy<_Predict>(predict), sp, this->x, this->P, gamma);
   
    // calcolo x e P (stime a priori) usando le stime di x attraverso sp
    unscent_transform(x1, tmp, sp, n, lambda, central_boost);
  }

 /// simulate the future state of the system (without change it)
 /// @param x1 return the future state
 /// @param input the input param
 void simulate(const boost::numeric::ublas::vector<T> & input, boost::numeric::ublas::vector<T> & x1)
 {
   // TODO
 }


/// use a linearized predict 
template<class _Predict>
void linear_predict(_Predict & predict)
 {
   const unsigned int n = this->size();
   const double k = kappa(n);
    
   kalman_filter<T, _matP>::predict(predict);
   // generate sigma_points used in observe
   sigma_points(_predict_policy<_Predict>(predict), sp, this->x, this->P, gamma );
 }

/// Uses an unscent transforms to predict the new state
/// predict must have a predict.f(x) function e predict.Q(x) function
template<class _Predict>
void predict(_Predict & predict)
 {
    const unsigned int n = this->size();
//     const double k = kappa(n);
    
    if(debug_kalman)
      std::cerr << "P-:" << this->P << std::endl;

    // segnalo un nuovo stato per permettere l'aggiornamento di Q
    predict.update(this->x);
    // NOTE aggiungo il rumore a P qua 
    this->P += predict.Q;
    // TODO serve lo stato aumentato n+q e nel nostro caso il rumore di processo e' uguale allo stato, percio' 2n
    //      vengono generati n sigma-points da P e n sigma points da Q
    // riempio sp con le proiezioni di x usando P
    sigma_points(_predict_policy<_Predict>(predict), sp, this->x, this->P, gamma);
   
    // calcolo x e P (stime a priori) usando le stime di x attraverso sp
    unscent_transform(this->x, this->P, sp, n, lambda, central_boost);
    
    if(debug_kalman)
      {
      std::cerr << "x-:" << this->x << std::endl;
      std::cerr << "P-:" << this->P << std::endl;
      }

 }

/// use a linearized observe
template<class _Observe>
void linear_observe(_Observe & observe)
 {
   kalman_filter<T, _matP>::observe(observe);
 }

template<class _Observe, class _Compare>
void observe(_Observe & observe, const boost::numeric::ublas::vector<T> & z, _Compare comp)
 {
    const unsigned int n = this->size();
//     const double k = kappa(n);

    // segnalo un nuovo x all'osservatore
    observe.update(this->x);
    
    // proietto sp su yp usando policy 2*n+1 punti
    project(_observe_policy<_Observe>(observe), yp, sp, 2*n+1);
    
    // calcolo y da yp
    average(y, yp, n, lambda);

    if(debug_kalman)
      {
      std::cerr << "y:" << y << std::endl;
      }
    
    // calcolo la covarianza Pyy e la crosscovarianza Pxy
    covariance(Pxy, Pyy, y, yp, this->x, sp, n, lambda, central_boost);

    if(debug_kalman)
      {
      std::cerr << "Pyy:" << Pyy << std::endl;
      std::cerr << "Pxy:" << Pxy << std::endl;
      }
    
    // fase di aggiornamento della misura
    Pyy += observe.R;
    
    // guadagno di Kalman
    boost::numeric::ublas::matrix<T> I_Pyy(Pyy.size1(), Pyy.size1());
    invert(Pyy, I_Pyy);
    this->K = boost::numeric::ublas::prod(Pxy, I_Pyy);

    if(debug_kalman)
      {
      std::cerr << "z:" << z << std::endl;
      std::cerr << "K:" << this->K << std::endl;
      }
    
    // aggiornamento stato 
    boost::numeric::ublas::vector<T> e = comp(z, y);
    if(debug_kalman)
      std::cerr << "e:" << e << std::endl;
    this->x += boost::numeric::ublas::prod(this->K, e );
    if(debug_kalman)
       std::cerr << "x+:" << this->x << std::endl;
    // aggiornamento varianza
     boost::numeric::ublas::matrix<T> Pyy_KT;
    Pyy_KT = boost::numeric::ublas::prod(Pyy, boost::numeric::ublas::trans(this->K) );
    this->P -= boost::numeric::ublas::prod(this->K, Pyy_KT);
  
 }
  
};

/// default compare: linear compare
template<class T>
class linear_compare {
public:
  boost::numeric::ublas::vector<T> operator()(const boost::numeric::ublas::vector<T> & z, const boost::numeric::ublas::vector<T> & s)
  {
    return z - s;
  }
};

/// A general purpose linear (o linearized) predict model
/// x_{k+1} = A * x_{k} + B * u_k + noise(Q_k)
/// @param _matA evolution matrix type (normally boost::numeric::ublas::matrix \< double \> )
/// @param _matB input matrix type (normally boost::numeric::ublas::matrix \< double \> )
/// @param _matQ process noise matrix type (normally boost::numeric::ublas::matrix \< double \> )
template<class _matA = boost::numeric::ublas::matrix<double>, class _matB = boost::numeric::ublas::matrix<double> , class _matQ = boost::numeric::ublas::matrix<double> >
struct linearized_predict {
  /// State transition matrix
  _matA A; 
  /// input state matrix
  _matB B;
  /// process noise covariance matrix
  _matQ Q;
  
  linearized_predict(int size) : A(size,size), B(size, size), Q(size,size) { }
};

/// Unscented predict model
/// x_{k+1} = f(x_k, u_k, w_k)
template<class _matQ = boost::numeric::ublas::matrix<double> >
struct unscented_predict {
  /// process noise covariance matrix
  _matQ Q;
  
  unscented_predict(int size) : Q(size, size) { }
  
};

/// A general purpose linear (o linearized) observe model
/// z_k = H_k * x_k + noise(R_k)
/// @param _matH observation matrix type (normally boost::numeric::ublas::matrix \< double \> )
/// @param _matR observation noise matrix type (normally boost::numeric::ublas::matrix \< double \> )
template<class _matH = boost::numeric::ublas::matrix<double> , class _matR = boost::numeric::ublas::matrix<double> >
struct linearized_observe {
  /// observation matrix
  _matH H;
  /// observation noise covariance matrix
  _matR R;
  
  linearized_observe(int state_size, int observation_size) : H(observation_size, state_size), R(observation_size, observation_size) { }
  
};

/// Unscented observe model
/// z_k = h(x_k, v_k)
template<class _matR = boost::numeric::ublas::matrix<double> >
struct unscented_observe {
  /// process noise covariance matrix
  _matR R;
  
  unscented_observe( int observation_size) : R(observation_size,observation_size) { }
  
};



#endif

